% Written by Tobias Grafke (grafke@cims.nyu.edu), based on the
% publication
%
%    T. Grafke, T. Schäfer, and E. Vanden-Eijnden,
%    "Long-Lasting Effects of Small Random Perturbations on Dynamical
%       Systems: Theoretical and Computational Tools" (2016)
%
% available at http://arxiv.org/abs/1604.03818.
%
% Computes the heteroclinic orbit via the string method.
%
% Usage:
%  x = string(x_initial, params)
%
% Returns the string x, which is the heteroclinic orbit connecting
% initial and final point through the saddle, which fulfills
% dx/dt||H_p(x,0). Here, params is a data structure containing the
% fields
%  params.H_p:        The p-derivative of the Hamiltonian as a
%                     function handle of x and p
%  params.iterations: (optional) Total number of iterations
%  params.plotevery:  (optional) Do a diagnostic plot every <plotevery>
%                     iterations
%  params.epsilon:    (optional) Relaxation step length
%  params.{xa,xb,xs}: (optional) initial, final and saddle point
%                     for diagnostic plotting purposes

function x = string(x_initial, params)
   % default params
   if isfield(params, 'iterations')
       iterations = params.iterations;
   else
       iterations = 1000;
   end

   if isfield(params, 'plotevery')
       plotevery = params.plotevery;
   else
       plotevery = 100;
   end

   if isfield(params, 'epsilon')
       epsilon = params.epsilon;
   else
       epsilon = 1e-1;
   end

   if isfield(params, 'xs')
       xsaddle = params.xs
   else
       xsaddle = []
   end

   x = x_initial;
   xa = x(1,:); xb = x(end,:);
   H_p = params.H_p;
   [Nt, Nx] = size(x);
   s = linspace(0,1,Nt);

   % reparametrize to standard arclength
   alpha = [0; cumsum(sqrt(sum((x(2:end,:)-x(1:end-1,:)).^2, 2)))];
   alpha = alpha/alpha(end);
   x = interp1(alpha, x, s, 'linear');

   for i = 1:iterations
       x = x + epsilon*H_p(x,0*x);

       % reset initial and final points to allow for string computation
       % between points that are not stable fixed points
       x(1,:) = xa; x(end,:) = xb;

       % reparametrize to standard arclength
       alpha = [0; cumsum(sqrt(sum((x(2:end,:)-x(1:end-1,:)).^2, 2)))];
       alpha = alpha/alpha(end);
       x = interp1(alpha, x, s, 'linear');

       % diagnostic plots
       if mod(i,plotevery)==0
           if (Nx == 2)
               figure(1)
               plot(x(:,1),x(:,2),'x-')
               if xsaddle
                   hold on
                   plot([xa(1),xsaddle(1),xb(1)], [xa(2),xsaddle(2),xb(2)], ...
                        'ro')
                   hold off
               end
               title('Trajectory')
               xlabel('x')
               ylabel('y')
               drawnow
           else
               figure(1)
               contourf(1:Nx, s, x, 25)
               title('String')
               xlabel('x')
               ylabel('s')

               b = H_p(x, 0*x);
               b2 = sum(b.^2, 2);

               figure(2)
               plot(s, b2, 'x-')
               title('Magnitude of drift')
               xlabel('s')
               ylabel('\sum b(x)^2')
               drawnow
           end
       end
   end
end
