% Written by Tobias Grafke (grafke@cims.nyu.edu), based on the
% publication
%
%    T. Grafke, T. Schäfer, and E. Vanden-Eijnden,
%    "Long-Lasting Effects of Small Random Perturbations on Dynamical
%       Systems: Theoretical and Computational Tools" (2016)
%
% available at http://arxiv.org/abs/1604.03818.
%
% Maier-Stein non-gradient test problem, defined by
%
%  du = (u-u^3 - \beta u v^2)dt + \sqrt{\epsilon} dW_u
%  dv = -(1+u^2)v dt + \sqrt{\epsilon} dW_v
%
% where \beta is a parameter, and the system is in detailed balance
% if \beta=1

Nt = 2^10;
Nx = 2;

% model parameters
beta = 10;
params.H = @(x,p) sum([x(:,1)-x(:,1).^3-beta*x(:,1).*x(:,2).^2, ...
                    -(1+x(:,1).^2).*x(:,2)].*[p(:,1),p(:,2)],2) + ...
                  0.5*(p(:,1).^2+p(:,2).^2);

params.H_x = @(x,p) [(1-3*x(:,1).^2-beta*x(:,2).^2).*p(:,1) + ...
                    (-2*x(:,1).*x(:,2)).*p(:,2), ...
                    (-2*beta*x(:,1).*x(:,2)).*p(:,1)-(1+x(:,1).^2).*p(:,2)];

params.H_p = @(x,p) [x(:,1)-x(:,1).^3-beta*x(:,1).*x(:,2).^2, ...
                    -(1+x(:,1).^2).*x(:,2)] + [p(:,1),p(:,2)];

params.invH_pp = @(x,p) 0*x+1;

% setup
s = linspace(0,1,Nt);
ds = s(2)-s(1);
Xs = fsolve(@(x) params.H_p(x,0*x),[-1,0]);
params.xa=Xs;
params.xb=-Xs;
params.xs = fsolve(@(x) params.H_p(x,0*x), 0.5*(params.xa+params.xb));

% initial trajectory
x_initial = [(params.xb(1)-params.xa(1))*s+params.xa(1); ...
             (params.xb(2)-params.xa(2))*s+params.xa(2) + 0.5*s.*(1-s)]';

disp('%%%%%%%%%%%%%%%%')
disp('String')
disp('%%%%%%%%%%%%%%%%')

params.epsilon = 5e-1;
params.iterations = 5e2;
params.plotevery = 100;

x_string = string(x_initial, params);

disp('%%%%%%%%%%%%%%%%')
disp('Minimizer')
disp('%%%%%%%%%%%%%%%%')

% restart initial trajectory

x_initial = [(params.xb(1)-params.xa(1))*s+params.xa(1); ...
             (params.xb(2)-params.xa(2))*s+params.xa(2) + 0.1*s.*(1-s)]';
params.iterations = 1e2;
params.plotevery = 10;
params.epsilon = 1e-1;

x_minimizer = sgmam(x_initial, params);

figure(1)
plot(x_string(:,1), x_string(:,2), 'x-g', ...
     x_minimizer(:,1), x_minimizer(:,2), 'x-b', ...
     [params.xa(1),params.xs(1),params.xb(1)], ...
     [params.xa(2),params.xs(2),params.xb(2)], ...
     'ro')
title('Overview')
legend('String', 'Minimizer')