% Written by Tobias Grafke (grafke@cims.nyu.edu), based on the
% publication
%
%    T. Grafke, T. Schäfer, and E. Vanden-Eijnden,
%    "Long-Lasting Effects of Small Random Perturbations on Dynamical
%       Systems: Theoretical and Computational Tools" (2016)
%
% available at http://arxiv.org/abs/1604.03818.
%
% Allen-Cahn/Cahn-Hilliard toy problem, defined by
%
%  d\phi = (1/\alpha Q (\phi-\phi.^3) - (\phi-\phi*)) dt + \sqrt{\epsilon} dW
%
% in component-wise notation, Q=[[1,-1];[-1,1]], and \phi* describes
% the tilt of the Allen-Cahn potential that leads to a disconnection
% of the slow manifold

Nt = 2^10;
Nx = 2;

% model parameters
alph = 0.01;
phiStar = [0.1,-0.1];

params.H = @(x,p) ((-x(:,2)+x(:,2).^3+x(:,1)-x(:,1).^3-...
                    alph*x(:,1)+phiStar(1)).*p(:,1) + 0.5*p(:,1).^2 ...
                   + (-x(:,1)+x(:,1).^3+x(:,2)-x(:,2).^3-...
                      alph*x(:,2)+phiStar(2)).*p(:,2) +0.5*p(:,2).^2);

params.H_x = @(x,p) [(1-3*x(:,1).^2-alph).*p(:,1)+(-1+3*x(:,1).^2).*p(:,2), ...
              (-1+3*x(:,2).^2).*p(:,1)+(1-3*x(:,2).^2-alph).*p(:,2)]

params.H_p = @(x,p) [-x(:,2)+x(:,2).^3+x(:,1)-x(:,1).^3-...
                    alph*x(:,1)+phiStar(1) + p(:,1), ...
                    -x(:,1)+x(:,1).^3+x(:,2)-x(:,2).^3-...
                    alph*x(:,2)+phiStar(2) + p(:,2)];

params.invH_pp = @(x,p) [0*x(:,1)+1, 0*x(:,1)+1];

% setup
s = linspace(0,1,Nt);
ds = s(2)-s(1);
params.xa = fsolve(@(x) params.H_p(x,0*x),[1,-1]);
params.xb = fsolve(@(x) params.H_p(x,0*x),[-1,1]);
params.xs = fsolve(@(x) params.H_p(x,0*x), 0.5*(params.xa+params.xb));

% initial trajectory
x_initial = [(params.xb(1)-params.xa(1))*s+params.xa(1); ...
             (params.xb(2)-params.xa(2))*s+params.xa(2) + 0.1*s.*(1-s)]';

disp('%%%%%%%%%%%%%%%%')
disp('String')
disp('%%%%%%%%%%%%%%%%')

params.epsilon = 1e-1;
params.iterations = 5e2;
params.plotevery = 100;

x_string = string(x_initial, params);

disp('%%%%%%%%%%%%%%%%')
disp('Minimizer')
disp('%%%%%%%%%%%%%%%%')

% start of with the string

params.iterations = 1e3;
params.plotevery = 50;
params.epsilon = 1e-1;

x_initial = x_string;
x_minimizer_forward = sgmam(x_initial, params);
x_initial = x_string(end:-1:1,:);
x_minimizer_backward = sgmam(x_initial, params);

figure(1)
plot(x_string(:,1), x_string(:,2), 'x-g', ...
     x_minimizer_forward(:,1), x_minimizer_forward(:,2), 'x-b', ...
     x_minimizer_backward(:,1), x_minimizer_backward(:,2), 'x-r', ...
     [params.xa(1),params.xs(1),params.xb(1)], ...
     [params.xa(2),params.xs(2),params.xb(2)], ...
     'ro')
title('Overview')
legend('String', 'Minimizer (forward)', 'Minimizer (backward)')